#!/usr/bin/env python3
"""
Script to extract unique values from ef1-ef12 columns in sub.csv
and append them to urls.txt
"""

import csv
import os

def extract_unique_values(csv_file, output_file):
    """
    Read CSV file and extract unique values from ef1 to ef12 columns
    
    Args:
        csv_file: Path to input CSV file
        output_file: Path to output text file
    """
    unique_values = set()
    
    # Read existing values from urls.txt if it exists
    if os.path.exists(output_file):
        with open(output_file, 'r', encoding='utf-8') as f:
            existing_values = set(line.strip() for line in f if line.strip())
            unique_values.update(existing_values)
    
    # Read CSV and extract values from ef1 to ef12
    with open(csv_file, 'r', encoding='utf-8') as f:
        reader = csv.DictReader(f)
        
        for row in reader:
            # Extract values from ef1 to ef12
            for i in range(1, 13):
                col_name = f'ef{i}'
                value = row.get(col_name, '').strip()
                
                # Only add non-empty values
                if value:
                    unique_values.add(value)
    
    # Write all unique values to file
    with open(output_file, 'w', encoding='utf-8') as f:
        for value in sorted(unique_values):
            f.write(value + '\n')
    
    print(f"Extracted {len(unique_values)} unique values from ef1-ef12 columns")
    print(f"Results written to {output_file}")

if __name__ == "__main__":
    csv_file = "./sub.csv"
    output_file = "urls.txt"
    
    if not os.path.exists(csv_file):
        print(f"Error: {csv_file} not found!")
        exit(1)
    
    extract_unique_values(csv_file, output_file)
