// Chat Application for Dr. Rajesh Kulkarni - Hair Loss Specialist

const OPENROUTER_API_URL = 'https://openrouter.ai/api/v1/chat/completions';
const MODEL = 'google/gemini-2.0-flash-001';
const API_KEY = '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';

const SYSTEM_PROMPT = `You are Dr. Rajesh Kulkarni's AI assistant at Keshaveda Hair Clinic in Mumbai. You are a knowledgeable and empathetic hair loss specialist assistant.

Your expertise covers:
- Hair loss causes (androgenetic alopecia, telogen effluvium, alopecia areata, etc.)
- Hair loss treatments (minoxidil, finasteride, PRP therapy, hair transplants)
- Scalp conditions and their treatments
- Hair care advice and nutrition for hair health
- Pre and post treatment care
- General information about Dr. Kulkarni's services

Guidelines:
1. Be warm, professional, and empathetic in your responses
2. Provide helpful, accurate information about hair loss and treatments
3. Always recommend consulting with Dr. Kulkarni in person for proper diagnosis
4. If someone asks a question NOT related to hair, hair loss, scalp health, or the clinic's services, politely decline and redirect them to hair-related topics. Say something like: "I appreciate your question, but I specialize only in hair loss and scalp health topics. Is there anything related to hair care or our treatments I can help you with?"
5. Keep responses concise but informative
6. Never provide specific medical prescriptions - always recommend an in-person consultation

Remember: You MUST refuse to answer questions about politics, general health unrelated to hair, entertainment, sports, or any other off-topic subjects.`;

// State
let messages = [];
let isRecording = false;
let recognition = null;
let currentUtterance = null;
let currentSpeakingButton = null;

// DOM Elements
const chatMessages = document.getElementById('chatMessages');
const messageInput = document.getElementById('messageInput');
const sendBtn = document.getElementById('sendBtn');
const voiceBtn = document.getElementById('voiceBtn');
const ttsEnabled = document.getElementById('ttsEnabled');
const recordingStatus = document.getElementById('recordingStatus');

// Initialize
function init() {
    setupEventListeners();
    setupSpeechRecognition();
    autoResizeTextarea();
}

// Event Listeners
function setupEventListeners() {
    sendBtn.addEventListener('click', sendMessage);

    messageInput.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            e.preventDefault();
            sendMessage();
        }
    });

    messageInput.addEventListener('input', autoResizeTextarea);

    voiceBtn.addEventListener('click', toggleVoiceInput);
}

// Auto-resize textarea
function autoResizeTextarea() {
    messageInput.style.height = 'auto';
    messageInput.style.height = Math.min(messageInput.scrollHeight, 120) + 'px';
}

// Speech Recognition Setup
function setupSpeechRecognition() {
    if ('webkitSpeechRecognition' in window || 'SpeechRecognition' in window) {
        const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
        recognition = new SpeechRecognition();
        recognition.continuous = false;
        recognition.interimResults = true;
        recognition.lang = 'en-US';

        recognition.onstart = () => {
            isRecording = true;
            voiceBtn.classList.add('recording');
            recordingStatus.textContent = 'Listening...';
        };

        recognition.onresult = (event) => {
            const transcript = Array.from(event.results)
                .map(result => result[0].transcript)
                .join('');
            messageInput.value = transcript;
            autoResizeTextarea();
        };

        recognition.onend = () => {
            isRecording = false;
            voiceBtn.classList.remove('recording');
            recordingStatus.textContent = '';
        };

        recognition.onerror = (event) => {
            console.error('Speech recognition error:', event.error);
            isRecording = false;
            voiceBtn.classList.remove('recording');

            // Provide user-friendly error messages
            let errorMsg = '';
            switch (event.error) {
                case 'not-allowed':
                    errorMsg = 'Microphone access denied. Please allow microphone access.';
                    break;
                case 'no-speech':
                    errorMsg = 'No speech detected. Please try again.';
                    break;
                case 'audio-capture':
                    errorMsg = 'No microphone found. Please check your device.';
                    break;
                case 'network':
                    errorMsg = 'Network error. Speech recognition requires internet.';
                    break;
                case 'aborted':
                    errorMsg = ''; // User aborted, no message needed
                    break;
                default:
                    errorMsg = 'Speech recognition error. Please try again.';
            }

            recordingStatus.textContent = errorMsg;
            if (errorMsg) {
                setTimeout(() => {
                    recordingStatus.textContent = '';
                }, 3000);
            }
        };
    } else {
        voiceBtn.style.display = 'none';
        console.warn('Speech recognition not supported in this browser');
    }
}

// Toggle Voice Input
function toggleVoiceInput() {
    if (!recognition) {
        recordingStatus.textContent = 'Speech recognition not supported';
        setTimeout(() => {
            recordingStatus.textContent = '';
        }, 3000);
        return;
    }

    // Check if we're on HTTPS (required for speech recognition in Chrome)
    if (location.protocol !== 'https:' && location.hostname !== 'localhost' && location.hostname !== '127.0.0.1') {
        recordingStatus.textContent = 'HTTPS required for voice input';
        setTimeout(() => {
            recordingStatus.textContent = '';
        }, 3000);
        return;
    }

    if (isRecording) {
        recognition.stop();
    } else {
        try {
            recognition.start();
        } catch (e) {
            // Can happen if recognition is already started
            console.error('Failed to start recognition:', e);
            recordingStatus.textContent = 'Voice input unavailable';
            setTimeout(() => {
                recordingStatus.textContent = '';
            }, 3000);
        }
    }
}

// Text-to-Speech using Web Speech API
function speak(text, button = null) {
    // Check if TTS is enabled (for auto-speak on new messages)
    if (!button && !ttsEnabled.checked) {
        return;
    }

    // Check if speech synthesis is supported
    if (!('speechSynthesis' in window)) {
        console.warn('Text-to-speech not supported in this browser');
        return;
    }

    // If clicking the same button that's currently speaking, stop it
    if (button && currentSpeakingButton === button && speechSynthesis.speaking) {
        stopSpeaking();
        return;
    }

    // Stop any current speech
    stopSpeaking();

    // Create new utterance
    currentUtterance = new SpeechSynthesisUtterance(text);
    currentUtterance.rate = 1.0;
    currentUtterance.pitch = 1.0;
    currentUtterance.volume = 1.0;

    // Try to use a natural voice
    const voices = speechSynthesis.getVoices();
    const preferredVoice = voices.find(v =>
        v.lang.startsWith('en') && (v.name.includes('Google') || v.name.includes('Natural'))
    ) || voices.find(v => v.lang.startsWith('en'));

    if (preferredVoice) {
        currentUtterance.voice = preferredVoice;
    }

    // Update button state
    if (button) {
        currentSpeakingButton = button;
        button.classList.add('speaking');
        button.title = 'Stop speaking';
    }

    // Handle speech end
    currentUtterance.onend = () => {
        resetSpeakingState();
    };

    currentUtterance.onerror = (e) => {
        console.error('Speech synthesis error:', e);
        resetSpeakingState();
    };

    // Start speaking
    speechSynthesis.speak(currentUtterance);
}

function stopSpeaking() {
    if (speechSynthesis.speaking) {
        speechSynthesis.cancel();
    }
    resetSpeakingState();
}

function resetSpeakingState() {
    if (currentSpeakingButton) {
        currentSpeakingButton.classList.remove('speaking');
        currentSpeakingButton.title = 'Listen to message';
    }
    currentSpeakingButton = null;
    currentUtterance = null;
}

// Add message to chat
function addMessage(content, role) {
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${role}`;

    const avatar = document.createElement('div');
    avatar.className = 'message-avatar';
    avatar.textContent = role === 'assistant' ? 'DR' : 'You';

    const contentDiv = document.createElement('div');
    contentDiv.className = 'message-content';
    contentDiv.innerHTML = `<p>${escapeHtml(content)}</p>`;

    // Add TTS button for assistant messages
    if (role === 'assistant') {
        const ttsBtn = document.createElement('button');
        ttsBtn.className = 'message-tts-btn';
        ttsBtn.title = 'Listen to message';
        ttsBtn.innerHTML = `
            <svg class="tts-icon-speaker" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <polygon points="11 5 6 9 2 9 2 15 6 15 11 19 11 5"></polygon>
                <path d="M15.54 8.46a5 5 0 0 1 0 7.07"></path>
                <path d="M19.07 4.93a10 10 0 0 1 0 14.14"></path>
            </svg>
            <svg class="tts-icon-stop" viewBox="0 0 24 24" fill="currentColor">
                <rect x="6" y="6" width="12" height="12" rx="2"></rect>
            </svg>
        `;
        ttsBtn.addEventListener('click', () => speak(content, ttsBtn));
        contentDiv.appendChild(ttsBtn);
    }

    messageDiv.appendChild(avatar);
    messageDiv.appendChild(contentDiv);
    chatMessages.appendChild(messageDiv);

    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Show typing indicator
function showTypingIndicator() {
    const indicator = document.createElement('div');
    indicator.className = 'message assistant';
    indicator.id = 'typing-indicator';
    indicator.innerHTML = `
        <div class="message-avatar">DR</div>
        <div class="message-content">
            <div class="typing-indicator">
                <span></span>
                <span></span>
                <span></span>
            </div>
        </div>
    `;
    chatMessages.appendChild(indicator);
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Remove typing indicator
function removeTypingIndicator() {
    const indicator = document.getElementById('typing-indicator');
    if (indicator) {
        indicator.remove();
    }
}

// Escape HTML to prevent XSS
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Get last N messages for context
function getRecentMessages(n = 5) {
    return messages.slice(-n);
}

// Send message
async function sendMessage() {
    const content = messageInput.value.trim();
    if (!content) return;

    // Add user message to UI and state
    addMessage(content, 'user');
    messages.push({ role: 'user', content });

    // Clear input
    messageInput.value = '';
    autoResizeTextarea();

    // Disable send button
    sendBtn.disabled = true;
    showTypingIndicator();

    try {
        // Build messages array with system prompt and last 5 messages
        const apiMessages = [
            { role: 'system', content: SYSTEM_PROMPT },
            ...getRecentMessages(5)
        ];

        const response = await fetch(OPENROUTER_API_URL, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${API_KEY}`,
                'HTTP-Referer': window.location.href,
                'X-Title': 'Keshaveda Hair Clinic Chat'
            },
            body: JSON.stringify({
                model: MODEL,
                messages: apiMessages,
                temperature: 0.7,
                max_tokens: 500
            })
        });

        if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error?.message || 'API request failed');
        }

        const data = await response.json();
        const assistantMessage = data.choices[0]?.message?.content || 'I apologize, but I could not generate a response. Please try again.';

        // Add assistant message to UI and state
        removeTypingIndicator();
        addMessage(assistantMessage, 'assistant');
        messages.push({ role: 'assistant', content: assistantMessage });

        // Speak the response
        speak(assistantMessage);

    } catch (error) {
        console.error('Error:', error);
        removeTypingIndicator();
        addMessage(`Sorry, there was an error: ${error.message}. Please try again.`, 'assistant');
    }

    sendBtn.disabled = false;
}

// Initialize app
init();
